### Directory Structure

Here's a suggested directory structure:

```
api-app/
│
├── config/
│   └── database.php
│
├── controllers/
│   ├── AuthController.php
│   ├── NewsController.php
│   ├── UserController.php
│   └── NotificationController.php
│
├── models/
│   ├── UserModel.php
│   ├── NewsModel.php
│   └── OtpModel.php
│
└── api/
    ├── register.php
    ├── login.php
    ├── update_profile.php
    ├── send_push_notification.php
    ├── verify_otp.php
    └── upload_foto_profil.php
```

### Configuration File

**config/database.php**

```php
<?php
$servername = "localhost";
$username = "your_username";
$password = "your_password";
$dbname = "your_database_name";

function getDbConnection() {
    global $servername, $username, $password, $dbname;
    $conn = new mysqli($servername, $username, $password, $dbname);
    
    if ($conn->connect_error) {
        die("Connection failed: " . $conn->connect_error);
    }
    
    return $conn;
}
?>
```

### Models

**models/UserModel.php**

```php
<?php
require_once '../config/database.php';

class UserModel {
    private $conn;

    public function __construct() {
        $this->conn = getDbConnection();
    }

    public function findUserById($id) {
        $stmt = $this->conn->prepare("SELECT * FROM user WHERE id = ?");
        $stmt->bind_param("i", $id);
        $stmt->execute();
        return $stmt->get_result()->fetch_assoc();
    }

    public function updateUserProfile($data) {
        $stmt = $this->conn->prepare("UPDATE user SET nama_lengkap = ?, alamat = ?, provinsi = ?, kota = ?, kecamatan = ?, kelurahan = ? WHERE id = ?");
        $stmt->bind_param("ssssssi", $data['nama_lengkap'], $data['alamat'], $data['provinsi'], $data['kota'], $data['kecamatan'], $data['kelurahan'], $data['user_id']);
        return $stmt->execute();
    }

    // Other user-related methods...
}
?>
```

**models/OtpModel.php**

```php
<?php
require_once '../config/database.php';

class OtpModel {
    private $conn;

    public function __construct() {
        $this->conn = getDbConnection();
    }

    public function verifyOtp($phone, $otp, $purpose) {
        $stmt = $this->conn->prepare("SELECT * FROM otp_codes WHERE phone = ? AND otp = ? AND purpose = ? AND used = 0 AND expires_at > NOW()");
        $stmt->bind_param("sss", $phone, $otp, $purpose);
        $stmt->execute();
        return $stmt->get_result()->fetch_assoc();
    }

    public function markOtpAsUsed($id) {
        $stmt = $this->conn->prepare("UPDATE otp_codes SET used = 1 WHERE id = ?");
        $stmt->bind_param("i", $id);
        return $stmt->execute();
    }

    // Other OTP-related methods...
}
?>
```

### Controllers

**controllers/AuthController.php**

```php
<?php
require_once '../models/UserModel.php';
require_once '../models/OtpModel.php';

class AuthController {
    private $userModel;
    private $otpModel;

    public function __construct() {
        $this->userModel = new UserModel();
        $this->otpModel = new OtpModel();
    }

    public function register($data) {
        // Registration logic...
    }

    public function verifyOtp($phone, $otp, $purpose) {
        $otpRecord = $this->otpModel->verifyOtp($phone, $otp, $purpose);
        if ($otpRecord) {
            $this->otpModel->markOtpAsUsed($otpRecord['id']);
            return ['success' => true, 'message' => 'OTP verified successfully.'];
        }
        return ['success' => false, 'message' => 'Invalid OTP or expired.'];
    }

    // Other authentication-related methods...
}
?>
```

### API Endpoints

**api/register.php**

```php
<?php
header('Content-Type: application/json');
require_once '../controllers/AuthController.php';

$controller = new AuthController();

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $data = json_decode(file_get_contents('php://input'), true);
    $response = $controller->register($data);
    echo json_encode($response);
}
?>
```

**api/verify_otp.php**

```php
<?php
header('Content-Type: application/json');
require_once '../controllers/AuthController.php';

$controller = new AuthController();

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $data = json_decode(file_get_contents('php://input'), true);
    $response = $controller->verifyOtp($data['phone'], $data['otp'], $data['purpose']);
    echo json_encode($response);
}
?>
```

### Summary

1. **Configuration**: The database connection is centralized in `config/database.php`.
2. **Models**: Each model handles database interactions related to a specific entity (e.g., User, OTP).
3. **Controllers**: Each controller handles the business logic and interacts with the models.
4. **API Endpoints**: The API files are simplified to handle requests and responses, delegating logic to the controllers.

This structure improves maintainability, readability, and scalability of the codebase. You can continue to refactor other API files similarly by creating appropriate models and controllers.